' Audio1.TV CDN Client for Roku (BrightScript)
' Integrated from AgentCache SDK
' 
' Usage:
'   cdn = Audio1CDN()
'   streamUrl = cdn.getPlaylistUrl("video-123", "720p")
'   
'   ' Or use adaptive streaming with master playlist:
'   streamUrl = cdn.getMasterPlaylistUrl("video-123")

function Audio1CDN(baseUrl = "https://agentcache.ai" as String) as Object
    cdn = {
        baseUrl: baseUrl
        
        ' Get HLS playlist URL for specific quality
        getPlaylistUrl: function(jobId as String, quality = "720p" as String) as String
            return m.baseUrl + "/hls/" + jobId + "/" + quality + "/playlist.m3u8"
        end function
        
        ' Get master playlist for adaptive bitrate streaming
        getMasterPlaylistUrl: function(jobId as String) as String
            return m.baseUrl + "/hls/" + jobId + "/master.m3u8"
        end function
        
        ' Get direct segment URL
        getSegmentUrl: function(jobId as String, quality as String, segment as String) as String
            return m.baseUrl + "/hls/" + jobId + "/" + quality + "/" + segment
        end function
        
        ' Create video content object for Roku player
        createVideoContent: function(jobId as String, quality = "720p" as String, title = "" as String) as Object
            content = createObject("roAssociatedArray")
            content.url = m.getPlaylistUrl(jobId, quality)
            content.streamFormat = "hls"
            content.title = title
            content.playStart = 0
            return content
        end function
        
        ' Create adaptive video content with automatic quality selection
        createAdaptiveVideoContent: function(jobId as String, title = "" as String) as Object
            content = createObject("roAssociatedArray")
            content.url = m.getMasterPlaylistUrl(jobId)
            content.streamFormat = "hls"
            content.title = title
            content.playStart = 0
            content.adaptiveMinStartBitrate = 0
            content.adaptiveMaxStartBitrate = 0  ' Let player decide
            return content
        end function
        
        ' Create content from channel metadata (for Audio1 channels)
        createChannelContent: function(channel as Object) as Object
            content = createObject("roAssociatedArray")
            
            if channel.videoId <> invalid
                content.url = m.getMasterPlaylistUrl(channel.videoId)
                content.streamFormat = "hls"
            else if channel.streamUrl <> invalid
                content.url = channel.streamUrl
                content.streamFormat = "hls"
            end if
            
            content.title = channel.title
            content.description = channel.description
            content.hdPosterUrl = channel.posterUrl
            content.playStart = 0
            
            return content
        end function
    }
    
    return cdn
end function

' Quality constants matching API presets
function QualityPresets() as Object
    return {
        MOBILE: "360p"
        SD: "480p"
        HD: "720p"
        FULL_HD: "1080p"
    }
end function

' Detect optimal quality based on Roku device capability
function detectOptimalQuality() as String
    deviceInfo = createObject("roDeviceInfo")
    model = deviceInfo.getModel()
    
    ' Roku 4K and Ultra devices
    if model.instr("4K") > 0 or model.instr("Ultra") > 0 then
        return "1080p"
    end if
    
    ' Check actual display resolution
    displaySize = deviceInfo.getDisplaySize()
    if displaySize.w >= 1920 then
        return "1080p"
    else if displaySize.w >= 1280 then
        return "720p"
    else if displaySize.w >= 854 then
        return "480p"
    else
        return "360p"
    end if
end function

' Get connection-aware quality (for adaptive initial quality)
function getConnectionAwareQuality() as String
    deviceInfo = createObject("roDeviceInfo")
    connectionInfo = deviceInfo.getConnectionInfo()
    
    ' Check connection type
    if connectionInfo.type = "WiFiConnection" then
        ' WiFi - check link quality
        if connectionInfo.quality >= 80 then
            return detectOptimalQuality()  ' Use display-based quality
        else if connectionInfo.quality >= 50 then
            return "720p"
        else
            return "480p"
        end if
    else if connectionInfo.type = "WiredConnection" then
        ' Wired - assume good connection
        return detectOptimalQuality()
    else
        ' Unknown or mobile hotspot - be conservative
        return "480p"
    end if
end function
